// -*- c++ -*-
/*=========================================================================

  Program:   Visualization Toolkit
  Module:    $RCSfile: vtkNetCDFReader.h,v $

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

/*-------------------------------------------------------------------------
  Copyright 2009 Sandia Corporation.
  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
  the U.S. Government retains certain rights in this software.
-------------------------------------------------------------------------*/

#include "vtkFileSeriesWriter.h"

#include "vtkClientServerInterpreter.h"
#include "vtkClientServerStream.h"
#include "vtkDataObject.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkObjectFactory.h"
#include "vtkProcessModule.h"
#include "vtkStreamingDemandDrivenPipeline.h"

#include "vtkSmartPointer.h"

#include <vtkstd/string>

//=============================================================================
vtkCxxRevisionMacro(vtkFileSeriesWriter, "$Revision$");
vtkStandardNewMacro(vtkFileSeriesWriter);

vtkCxxSetObjectMacro(vtkFileSeriesWriter, Writer, vtkWriter);

//-----------------------------------------------------------------------------
vtkFileSeriesWriter::vtkFileSeriesWriter()
{
  this->FileName = NULL;
  this->Writer = NULL;
  this->FileNameMethod = NULL;
  this->SetFileNameMethod("SetFileName");
  this->WriteAllTimeSteps = 1;
  this->FileIndexSize = 4;

  this->NumberOfTimeSteps = 0;
  this->CurrentTimeIndex = 0;
}

vtkFileSeriesWriter::~vtkFileSeriesWriter()
{
  this->SetWriter(NULL);
}

void vtkFileSeriesWriter::PrintSelf(ostream &os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);

  os << indent << "FileName: " << this->FileName << endl;
  os << indent << "Writer: " << this->Writer << endl;
  os << indent << "FileNameMethod: " << this->FileNameMethod << endl;
  os << indent << "WriteAllTimeSteps: " << this->WriteAllTimeSteps << endl;
  os << indent << "FileIndexSize: " << this->FileIndexSize << endl;
}

//-----------------------------------------------------------------------------
int vtkFileSeriesWriter::ProcessRequest(vtkInformation *request,
                                        vtkInformationVector **inputVector,
                                        vtkInformationVector *outputVector)
{
  if (!this->Writer)
    {
    vtkErrorMacro(<< "ProcessRequest called before Writer set.");
    return 0;
    }

  if (request->Has(vtkDemandDrivenPipeline::REQUEST_INFORMATION()))
    {
    return this->RequestInformation(request, inputVector, outputVector);
    }

  if (request->Has(vtkStreamingDemandDrivenPipeline::REQUEST_UPDATE_EXTENT()))
    {
    return this->RequestUpdateExtent(request, inputVector, outputVector);
    }

  if (request->Has(vtkDemandDrivenPipeline::REQUEST_DATA()))
    {
    return this->RequestData(request, inputVector, outputVector);
    }

  return this->CallWriter(request, inputVector, outputVector);
}

//-----------------------------------------------------------------------------
int vtkFileSeriesWriter::RequestInformation(vtkInformation *request,
                                            vtkInformationVector **inputVector,
                                            vtkInformationVector *outputVector)
{
  if (!this->CallWriter(request, inputVector, outputVector)) return 0;

  vtkInformation *inInfo = inputVector[0]->GetInformationObject(0);
  this->NumberOfTimeSteps
    = inInfo->Length(vtkStreamingDemandDrivenPipeline::TIME_STEPS());

  return 1;
}

//-----------------------------------------------------------------------------
int vtkFileSeriesWriter::RequestUpdateExtent(vtkInformation *request,
                                             vtkInformationVector **inputVector,
                                             vtkInformationVector *outputVector)
{
  if (!this->CallWriter(request, inputVector, outputVector)) return 0;

  if (this->WriteAllTimeSteps && (this->NumberOfTimeSteps > 0))
    {
    vtkInformation *inInfo = inputVector[0]->GetInformationObject(0);
    double *timeSteps
      = inInfo->Get(vtkStreamingDemandDrivenPipeline::TIME_STEPS());
    inInfo->Set(vtkStreamingDemandDrivenPipeline::UPDATE_TIME_STEPS(),
                &(timeSteps[this->CurrentTimeIndex]), 1);
    }

  return 1;
}

//-----------------------------------------------------------------------------
int vtkFileSeriesWriter::RequestData(
                                 vtkInformation *vtkNotUsed(request),
                                 vtkInformationVector **inputVector,
                                 vtkInformationVector *vtkNotUsed(outputVector))
{
  if (!this->FileName)
    {
    vtkErrorMacro(<< "FileName not set.");
    return 0;
    }

  // Set filename in Writer.
  if (this->WriteAllTimeSteps && (this->NumberOfTimeSteps > 1))
    {
    char format[10];
    sprintf(format, "%%0%dd", this->FileIndexSize);
    char index[20];
    sprintf(index, format, this->CurrentTimeIndex);
    vtkstd::string fname = this->FileName;
    vtkstd::string::size_type extension = fname.rfind('.');
    fname.insert(extension, index);
    this->SetWriterFileName(fname.c_str());
    }
  else
    {
    this->SetWriterFileName(this->FileName);
    }

  // It would be nice to call RequestData directly on the Writer, but some
  // writers override the Write function expecting it to be called.  Thus, we
  // instead shallow copy the data, pass it to the writer, and let it go.  This
  // will probably cause other request like RequestInformation and
  // RequestUpdateExtent to be called on the writer.
  vtkDataObject *input = vtkDataObject::GetData(inputVector[0]);
  vtkSmartPointer<vtkDataObject> inputCopy;
  inputCopy.TakeReference(input->NewInstance());
  inputCopy->ShallowCopy(input);

  // Should I be using the vtkPVTrivalProducer here?
  this->Writer->SetInput(inputCopy);
  this->Writer->Write();

  return 1;
}

//-----------------------------------------------------------------------------
int vtkFileSeriesWriter::CallWriter(vtkInformation *request,
                                    vtkInformationVector **inputVector,
                                    vtkInformationVector *outputVector)
{
  return this->Writer->vtkAlgorithm::ProcessRequest(request,
                                                    inputVector, outputVector);
}

//-----------------------------------------------------------------------------
void vtkFileSeriesWriter::SetWriterFileName(const char* fname)
{
  if (this->Writer && fname)
    {
    vtkClientServerID csId = 
      vtkProcessModule::GetProcessModule()->GetIDFromObject(this->Writer);
    if (csId.ID && this->FileNameMethod)
      {
      vtkProcessModule* pm = vtkProcessModule::GetProcessModule();
      // Get the local process interpreter.
      vtkClientServerInterpreter* interp = pm->GetInterpreter();
      vtkClientServerStream stream;
      stream << vtkClientServerStream::Invoke
             << csId << this->FileNameMethod << fname
             << vtkClientServerStream::End;
      interp->ProcessStream(stream);
      }
    }
}
